<?php

namespace App\Http\Controllers\Api\User;

use App\Http\Controllers\Controller;
use App\Http\Controllers\Api\Traits\ApiResponse;
use App\Http\Resources\NotificationResource;
use Illuminate\Support\Facades\Auth;
use Illuminate\Http\Request;

class NotificationController extends Controller
{
    use ApiResponse;

    /**
     * @OA\Get(
     *     path="/api/user/notifications",
     *     summary="Get the user's notifications",
     *     tags={"Notifications"},
     *     security={{"bearerAuth":{}}},
     *     @OA\Parameter(
     *         name="Accept-Language",
     *         in="header",
     *         required=false,
     *         description="Language preference (e.g., 'en', 'ar')",
     *         @OA\Schema(type="string", enum={"en", "ar"}, default="en")
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="The user's notifications",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="data", type="array", @OA\Items(ref="#/components/schemas/NotificationResource")),
     *             @OA\Property(property="meta", type="object",
     *                 @OA\Property(property="current_page", type="integer", example=1),
     *                 @OA\Property(property="from", type="integer", example=1),
     *                 @OA\Property(property="last_page", type="integer", example=1),
     *                 @OA\Property(property="path", type="string", example="http://momento.test/api/user/notifications"),
     *                 @OA\Property(property="per_page", type="integer", example=10),
     *                 @OA\Property(property="to", type="integer", example=7),
     *                 @OA\Property(property="total", type="integer", example=7)
     *             ),
     *             @OA\Property(property="message", type="string", example="Notifications retrieved successfully.")
     *         )
     *     ),
     *     @OA\Response(
     *         response=401,
     *         description="Unauthenticated"
     *     )
     * )
     */
    public function index()
    {
        /** @var \App\Models\User $user */
        $user = Auth::user();
        $notifications = $user->notifications()->paginate(10);

        return $this->successResponse(NotificationResource::collection($notifications), __('messages.notifications_retrieved_successfully'));
    }

    /**
     * @OA\Post(
     *     path="/api/user/notifications/mark-all-as-read",
     *     summary="Mark all unread notifications as read",
     *     tags={"Notifications"},
     *     security={{"bearerAuth":{}}},
     *     @OA\Response(
     *         response=200,
     *         description="All notifications marked as read",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="data", type="object", @OA\Property(property="count", type="integer", example=5)),
     *             @OA\Property(property="message", type="string", example="All notifications marked as read.")
     *         )
     *     ),
     *     @OA\Response(
     *         response=401,
     *         description="Unauthenticated"
     *     )
     * )
     */
    public function markAllAsRead(Request $request)
    {
        $count = $request->user()->myNotifications()->whereNull('read_at')->update(['read_at' => now()]);
        return $this->successResponse(['count' => $count], __('messages.all_notifications_marked_as_read'));
    }

    /**
     * @OA\Post(
     *     path="/api/user/notifications/{id}/mark-as-read",
     *     summary="Mark a specific notification as read",
     *     tags={"Notifications"},
     *     security={{"bearerAuth":{}}},
     *     @OA\Parameter(
     *         name="id",
     *         in="path",
     *         required=true,
     *         description="The ID of the notification",
     *         @OA\Schema(type="string")
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Notification marked as read",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="data", ref="#/components/schemas/NotificationResource"),
     *             @OA\Property(property="message", type="string", example="Notification marked as read.")
     *         )
     *     ),
     *     @OA\Response(
     *         response=401,
     *         description="Unauthenticated"
     *     ),
     *     @OA\Response(
     *         response=404,
     *         description="Notification not found"
     *     )
     * )
     */
    public function markAsRead(Request $request, string $id)
    {
        $notification = $request->user()->notifications()->find($id);

        if (!$notification) {
            return $this->errorResponse(__('messages.notification_not_found'), 404);
        }

        if (is_null($notification->read_at)) {
            $notification->markAsRead();
        }

        return $this->successResponse(new NotificationResource($notification), __('messages.notification_marked_as_read'));
    }
}
