<?php

namespace App\Http\Controllers\Api;

use App\Http\Resources\SettingResource;
use App\Models\Setting;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log; // Added for error logging

class SettingController
{
    use Traits\ApiResponse;

    /**
     * @OA\Get(
     *     path="/api/settings",
     *     summary="Get application settings",
     *     tags={"Public"},
     *     @OA\Parameter(
     *         name="Accept-Language",
     *         in="header",
     *         required=false,
     *         description="Language preference (e.g., 'en', 'ar')",
     *         @OA\Schema(type="string", enum={"en", "ar"}, default="en")
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Settings retrieved successfully",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="data", ref="#/components/schemas/App_Http_Resources_Api_SettingResource"),
     *             @OA\Property(property="message", type="string", example="Settings retrieved successfully.")
     *         )
     *     ),
     *     @OA\Response(
     *         response=500,
     *         description="Unexpected error",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=false),
     *             @OA\Property(property="message", type="string", example="Unexpected error retrieving settings.")
     *         )
     *     )
     * )
     */
    public function index(Request $request): JsonResponse
    {
        try {
            $websiteSetting = Setting::where('key', 'website')->latest()->first();

            return $this->successResponse(new SettingResource($websiteSetting), __('messages.settings_retrieved_successfully'));
        } catch (\Exception $e) {
            Log::error("Setting index error: " . $e->getMessage());
            return $this->errorResponse(__('messages.unexpected_error_retrieving_settings'), 500);
        }
    }
}
