<?php

namespace App\Filament\Resources;

use App\Filament\Resources\ReviewResource\Pages;
use App\Filament\Resources\ServiceResource;
use App\Filament\Resources\PackageResource;
use App\Filament\Resources\UserResource;
use App\Models\Review;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Actions\DeleteAction;
use Filament\Tables\Actions\ViewAction;
use Filament\Tables\Actions\BulkActionGroup;
use Filament\Tables\Actions\DeleteBulkAction;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Filters\SelectFilter;
use Illuminate\Database\Eloquent\Builder;

class ReviewResource extends Resource
{
    protected static ?int $navigationSort = 6;

    public static function getNavigationGroup(): ?string
    {
        return __('Content Management');
    }
    public static function getNavigationBadge(): ?string
    {
        return static::getModel()::count();
    }
    public static function getLabel(): ?string
    {
        return __('Review');
    }

    public static function getPluralLabel(): ?string
    {
        return __('Reviews');
    }
    protected static ?string $model = Review::class;

    protected static ?string $navigationIcon = 'heroicon-o-star';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                //
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('id')
                    ->label(__('ID'))
                    ->sortable(),
                TextColumn::make('rating')
                    ->label(__('Rating'))
                    ->sortable(),
                TextColumn::make('comment')
                    ->label(__('Comment'))
                    ->limit(50),
                TextColumn::make('product.name_' . app()->getLocale())
                    ->label(__('Product'))
                    ->url(fn($record) => $record->product->type == 'service' ? ServiceResource::getUrl('view', ['record' => $record->product_id]) : PackageResource::getUrl('view', ['record' => $record->product_id]))
                    ->sortable(),
                TextColumn::make('user.first_name')
                    ->label(__('User'))
                    ->url(fn($record) => UserResource::getUrl('view', ['record' => $record->user_id]))
                    ->sortable(),
                TextColumn::make('created_at')
                    ->label(__('Created At'))
                    ->dateTime()
                    ->sortable(),
            ])
            ->filters([
                SelectFilter::make('product_id')
                    ->label(__('Product'))
                    ->relationship('product', 'name_' . app()->getLocale())
                    ->multiple()
                    ->searchable()
                    ->preload(),
                SelectFilter::make('rating')
                    ->label(__('app.filament.fields.rating'))
                    ->options([
                        '1' => '1 Star',
                        '2' => '2 Stars',
                        '3' => '3 Stars',
                        '4' => '4 Stars',
                        '5' => '5 Stars',
                    ])
                    ->multiple(),
                Filter::make('created_at')
                    ->form([
                        Forms\Components\DatePicker::make('created_from')->label(__('Created From')),
                        Forms\Components\DatePicker::make('created_until')->label(__('Created To')),
                    ])
                    ->query(function (Builder $query, array $data): Builder {
                        return $query
                            ->when(
                                $data['created_from'],
                                fn(Builder $query, $date): Builder => $query->whereDate('created_at', '>=', $date),
                            )
                            ->when(
                                $data['created_until'],
                                fn(Builder $query, $date): Builder => $query->whereDate('created_at', '<=', $date),
                            );
                    }),
            ])
            ->actions([
                ViewAction::make(),
                DeleteAction::make(),
            ])
            ->bulkActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                ]),
            ])
            ->modifyQueryUsing(fn(Builder $query) => $query->with(['product', 'user']));
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListReviews::route('/'),
            // 'view' => Pages\ViewReview::route('/{record}'),
        ];
    }

    public static function canCreate(): bool
    {
        return false;
    }

    public static function canEdit($record): bool
    {
        return false;
    }

    public static function canView($record): bool
    {
        return false;
    }
}
